% Function: This program uses the matrix of multiplex data points (such as 
% median reporter values, fold changes, p-values, etc) and returns a color
% graphical display figure with these values (Note that the function
% inverts the matrix during plotting)

% Inputs:
% Matrix - a (rows x col) matrix where the rows denote the probes and the
%    col denote the ips
% Title - a string that will be used for the title
% ProbeLabels - a cell that contains the labels of the probes 
%    ie ({'Probe1','Probe2',....})
% IPLabels = a cell that contains the labels of the ips 
%    ie ({'IP1','IP2',....})
% ColorBarScale - a boolean value used to determine if the color bar will
%    be on a log (0) or linear (1) scale
% ColorBarFile - the name of the txt file for the colorbar to be used in
%    the analysis (default is jet)
% 
% NOTE - both probe labels and ips must be in the same order as the data
% was analyzed previously

% Outputs:
% No function output, but this program will create a color graphical 
% display figure


function [] = PlotMatrixINV(MatrixPreInv,Title,ProbeLabels,IPLabels,...
    ColorBarScale, ColorMinMax, ColorBarFile)

Matrix = MatrixPreInv';

% Plots the matrix as a heat map
imagesc(Matrix);

% Determines the numerical value in each entry of the matrix and converts 
% and then stores the value in the variable heightStrings
% The format is customized based on the matrix being plotted (ie Colormap
% median fluorescence has 5 signicance values, FoldChange has 3 float
% points, etc)
if strcmp(ColorBarFile,'Colormap.txt') == 1;
    formatSpec = '%.5g';
elseif strcmp(ColorBarFile,'FoldChange.txt') == 1;
    formatSpec = '%.3f';
elseif strcmp(ColorBarFile,'LogFoldChangeRB.txt') == 1;
    formatSpec = '%.3f';
elseif strcmp(ColorBarFile,'PMap_441.txt') == 1;
    formatSpec = '%.2g';
else
    formatSpec = '%.3g';
end
heightStrings = strjust(num2str(Matrix(:),formatSpec),'center');

% Creates a meshgrid (ie a grid to match the matrix values) with the number
% of rows determined by the number of probes (probbe labels) and the number
% of ips (ip labels) determining the number of columns
[text_x,text_y] = meshgrid(1:length(ProbeLabels),1:length(IPLabels));

% Inserts the text (matrix entry values) into the heat map plot (with the
% data centered within each cell)
text(text_x(:),text_y(:),heightStrings,...
    'FontSize', 14,...
    'FontWeight', 'Bold', ...
    'HorizontalAlignment','Center')


% These next few lines of codes are used to replace the numerical values of
% the heat map with the probe and ip labels

% Creates xTicks and yTicks variables that are integers from 1 to the
% number of probe or ip labels (1,2,3,...length(probe/ip))
xTicks = 1:1:length(ProbeLabels);
yTicks = 1:1:length(IPLabels);

% These lines then ensure that the tick marks are defined as desired from
% the code above (1,2,3,...length(probe/ip)) and then replaced the values
% with the proper probe/ip labels
set(gca, 'XTick' , xTicks);
set(gca, 'XTickLabel', ProbeLabels);
set(gca, 'YTick', yTicks);
set(gca, 'YTickLabel', IPLabels);
set(gca, 'FontSize', 14); 
set(gca, 'CLim',[ColorMinMax(1),ColorMinMax(2)]);
set(gcf, 'Color','w');

% Removes the tick mark for aesthetic reasons
set(gca,'TickLength', [0,0])

% Formats the x,y axis labels 
xlabel('Probes', 'FontSize', 20)
ylabel('IPs', 'FontSize', 20)
% Formats the title
title(Title, 'FontSize', 24)

% adjustment for windows so that the colorbar can have more than 256 shades
set(gcf, 'Renderer', 'zbuffer')

% if then statement to determine if the color scale should be defined
% logarithmically or linearly

% ColorBarScale == 0, if this is true then the color bar is defined using 
% my custom log colorbar
if ColorBarScale == 0;
    
    % If/else statement to determine if the ColorBarFile is defined using a
    % saved txt file (if true it will use the saved color bar otherwise it
    % will default to jet)
    if (strcmp(ColorBarFile,'Colormap.txt') == 1 || ...
            strcmp(ColorBarFile,'FoldChange.txt') == 1 ||...
            strcmp(ColorBarFile,'PMap_441.txt') == 1||...
            strcmp(ColorBarFile,'LogFoldChangeRB.txt') == 1);
        % imports the custom log color bar that was saved as a txt value
        C = importdata(ColorBarFile);
        % defines the colormap using this custom log color scheme
        colormap(C);
    else
        colormap(jet(2048*50));
    end
    
    % defines the position of the colorbar
    %hC = colorbar('location', 'SouthOutside');
    % changes the scale of the color bar to a log scale
    %set(hC, 'XScale', 'log', 'FontSize', 16);

% checks to see if ColorBar == 1, if this is true then the color bar is
% defined using a standard color bar    
elseif (ColorBarScale == 1)
    
    % If/else statement to determine if the ColorBarFile is defined using a
    % saved txt file (if true it will use the saved color bar otherwise it
    % will default to jet)
    if (strcmp(ColorBarFile,'Colormap.txt') == 1 || ...
            strcmp(ColorBarFile,'FoldChange.txt') == 1 || ...
            strcmp(ColorBarFile,'PMap_441.txt') == 1||...
            strcmp(ColorBarFile,'LogFoldChangeRB.txt') == 1);
        % imports the custom log color bar that was saved as a txt value
        C = importdata(ColorBarFile);
        % defines the colormap using this custom log color scheme
        colormap(C);
    else
        colormap(jet(2048*50));
    end
    
    if strcmp(ColorBarFile,'PMap_441.txt') == 1;
        colorbar('location', 'SouthOutside', 'FontSize', 16,'XTick', ...
            [0,0.05,0.2,0.4,0.6,0.8,1], 'XTickLabel',...
            char('0','0.05','0.2','0.4','0.6','0.8','1'))
    else
        colorbar('location', 'SouthOutside', 'FontSize', 16)
    end
    
else

end




